;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

pro comps_cal_st_jr

;  the procedure to determine the CoMP-S polarimetric calibration - D matrix+effciencies

; v.2 - JR - 2022/12/19-2023/01/18 - added descriptions of the code,
;                                    tested matrix syntax + manipilation actions   
; v.1 - ST - 2013/02/17

; .comp comps_cal_st_jr
; comps_cal_st_jr
  
; NOTES:
;
save_jpg=1
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; COMMON block for I_OBS and WAVE wariables  
common calibrate,i_obs,wave

close,/all

; parameters
key_cal_data_source=1           ; HAO 2012/02/17
key_cal_data_source=2           ; LSO 2023/01/18 with 2022/08/04 pol cal data

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; ST - v.1 - reading the calibration data taken at HAO - from FITS files or from
;            SAV file

; ST - calibration files can be read in any order and the observations will be ordered

if key_cal_data_source eq 1 then begin ; HAO 2012/02/17

   ;openpost,dev,default='s'
   ans=' '

   nx=2560 & ny=2160

   debug='no'                   ;debug mode, 'yes' or 'no'
   read_in='no'

   in_disk='C:\Data' & out_disk='C:\Data'
   out_disk=in_disk

   date='20110311'
   infiles='cal_files.txt'      ;file with list of file names
   indir=in_disk+'\'+date+'\'
   outdir=out_disk+'\'+date+' Reduced\'
   openw,2,'cal.txt'            ;open log file

   ;  define subarray region and region outside it
   nsub=600                     ;size of subarray over which to average
   x=rebin(findgen(nx),nx,ny)-float(nx)/2.
   y=transpose(rebin(findgen(ny),ny,nx))-float(ny)/2.
   subarray=where( abs(x) lt nsub and abs(y) lt nsub, complement=outside)

   ;  read in data or restore?
   if read_in eq 'yes' then begin
      
      ;  open file containing filenames, read in filenames and associated wavelength regions
      openr,1,outdir+infiles
      str=' '
      nfile=0
      files=strarr(200)
      reg=fltarr(200)
      while not eof(1) do begin
         readf,1,format='(a32,f)',str,r
         files[nfile]=str
         reg[nfile]=r
         nfile=nfile+1
      endwhile
      close,1
      
      files=files[0:nfile-1]
      reg=reg[0:nfile-1]

      regions=reg(uniq(reg,sort(reg))) ;identify wavelength regions
      nreg=n_elements(regions)         ;number of wavelength regions
      print,nfile,' Calibration files in ',nreg,' regions'
      if debug eq 'yes' then print,regions
      
      ;  loop over files and compute observed intensities, place into obs(4,7,nreg)
      ;  in the order of state, for each wavelength region
      obs=fltarr(4,7,nreg)
      state=['CLEAR','I+Q','I-Q','I+U','I-U','I+V','I-V'];observations should put in this order

      for ireg=0,nreg-1 do begin
         
         print,regions[ireg]
         good=where(reg eq regions[ireg]) ;locate files within this region
         for ical=0,6 do begin

		name=indir+files[good[ical]]
		print,name
		fits_open,name,fcb       ;open input fits file
		num=fcb.nextend
		if debug eq 'yes' then print,num,' images in file'

                ;  take inventory of images in this file
                comps_inventory,fcb,beam,group,wave,mod_state,type,filter,expose,diffuser,$
                                cal_state,camera
		istate=where(state eq cal_state)	;find index of cal state

                ;  read header of first image and get time and exposure time
		fits_read,fcb,d,header,/header_only,exten_no=1
		date_str=sxpar(header,'DATE_OBS')
		time_str=sxpar(header,'TIME_OBS')
                exposure=sxpar(header,'EXPOSURE')
                
		hours=float(strmid(time_str,0,2))		;  interpret time
		mins=float(strmid(time_str,2,2))
		secs=float(strmid(time_str,4,6))

		time=hours+mins/60.+secs/3600.
		if debug eq 'yes' then print,time_str,hours,mins,secs,time

                ;  interpolate dark image for time of observation

		dark=dark_interp(outdir,time,exposure)
                if debug eq 'yes' then print,'read dark image at:',time,' with exposure of:',$
                                             exposure

                ;  read in and average intensity over a subarray
		for i=0,num-1 do begin
                   fits_read,fcb,dat,header,exten_no=i+1
                   dat=float(dat)-dark
                   imod=i mod 4
                   obs[imod,istate,ireg]=obs[imod,istate,ireg]+$
                                         mean(dat[subarray])-median(dat[outside])
                endfor
		fits_close,fcb

		obs[*,istate,ireg]=4.*obs[*,istate,ireg]/float(num)
                wait,0.01
                
             endfor

      endfor

      save,regions,obs

   endif else restore

endif

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; JR - v.2 - reading the pol calibration data taken at LSO - from SAV file

if key_cal_data_source eq 2 then begin ;  LSO 2012/08/04 pol cal data

   stat_new_cam_set:
   cam=0 & set=0
   print
   read,cam,set,prompt=' INPUT: camera (1,2), set (1,2) = '
   if (cam ne 2 and cam ne 1) or (set ne 2 and set ne 1) then goto,stat_new_cam_set
   
   fileadd='_cam'+strtrim(string(cam),2)+'_set'+strtrim(string(set),2)+'.sav'
   sav_filename_in='~/comp/lso_observer/data_reduction/actual/20220804_cali/calibration_data/'+$
                 'lso_comps_20220804_cali0Dst_reg0587'+fileadd
   restore,sav_filename_in,/ver
   
   ;help,regions,cal_meas_0D_st,cal_meas_0D_sub_st,n_pol,n_cal,cal_optics,x_sub1,x_sub2,y_sub1,y_sub2,date
   ;; REGIONS         FLOAT     = Array[1]     - just 587 nm
   ;; CAL_MEAS_0D_ST  FLOAT     = Array[4, 7]
   ;; CAL_MEAS_0D_SUB_ST
   ;; FLOAT     = Array[4, 7]
   ;; N_POL           INT       =        4
   ;; N_CAL           INT       =        7
   ;; CAL_OPTICS      STRING    = Array[7]
   ;; X_SUB1          INT       =      400     - central sub-aread of
   ;;                                            the 1280x1080 pixel chip 
   ;; X_SUB2          INT       =      599
   ;; Y_SUB1          INT       =      400
   ;; Y_SUB2          INT       =      599
   ;; DATE            STRING    = '20220804'
   ;print,regions,n_pol,n_cal,cal_optics
   ; 587.600
   ; 4
   ; 7
   ; CLEAR
   ; LIN POL    0deg, LIN POL  90 deg, LIN POL  45 deg, LIN POL 135 deg,
   ; LIN POL   0 deg + RETARDER  45 deg, LIN POL   0 deg + RETARDER 135 deg

   ; additions for opening the ASCII TXT output file
   fileadd1='_cam'+strtrim(string(cam),2)+'_set'+strtrim(string(set),2)
   openw,2,date+fileadd1+'_cal.log'            ;open log file
   dev='s'
   
   ; renaming
   obs=CAL_MEAS_0D_SUB_ST

   ;new variables
   date_str=strmid(sav_filename_in,strpos(sav_filename_in,'lso_comps_',/reverse_search)+10,8)
   set_str=strtrim(string(set),2)
   cam_str=strtrim(string(cam),2)
   
endif

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; ST code for determiantion of the Dmatrix, efficiencies + JR explanations

;  compute fit 

nreg=n_elements(regions) ; number of sp. lines

; variables for all lines/regions
ptrans=fltarr(nreg)
rtrans=fltarr(nreg)
delta=fltarr(nreg)
stokes=fltarr(4,nreg)
o_matrix=fltarr(4,4,nreg)
a_matrix=fltarr(4,4,nreg)
a_invert_matrix=fltarr(4,4,nreg)
d_matrix=fltarr(4,4,nreg)
chi2=fltarr(nreg)
efficiencies=fltarr(4,nreg)

; cycle of fitting+ detrmination of the D matrix for all lines/regions
for ireg=0,nreg-1 do begin

        ; common block variable 
	i_obs=obs[*,*,ireg] ; observer 4x7 pol cal intensities
        wave=regions[ireg]  ; line wavelength

        ; printing outputs
        print
        print,'Date: ',date
        print,'Camera: ',cam
        print,'Dataset: ',set
        printf,2
        printf,2,'Date: ',date
        printf,2,'Camera: ',cam
        printf,2,'Dataset: ',set
        
        print,'Line:',wave
	printf,2,'Line:',wave

        print
        printf,2
        print,'I_obs:'
        printf,2,'I_obs'
        print,i_obs
        printf,2,i_obs

        ; fiting the observed intensities for 4 mod. states and for 7 cal. states
        a=comps_fit_for_omx_st_jr()
        
        ; model intensities for the actual the best fit 
        model=call_function('fit_funct_st_jr',a)       

        print
        printf,2
        print,'Model:'
        printf,2,'Model:'
        print,model
        printf,2,model

        ; normalized intensity differencies: (obs-model)/obs
        i_obs_model_diff_norm=(i_obs-model)/i_obs

        print
        printf,2
        print,'I_Obs_Model_Diff_Norm: i_obs_model_diff_norm=(i_obs-model)/i_obs'
        printf,2,'I_Obs_Model_Diff_Norm: i_obs_model_diff_norm=(i_obs-model)/i_obs'
        print,i_obs_model_diff_norm
        printf,2,i_obs_model_diff_norm
        
        ; variable renaming
	ptrans[ireg]=a[0]
	rtrans[ireg]=a[1]
	delta [ireg]=a[2]
        stokes[*,ireg]=a[3:6]

        print
        print,'Polarizer trans, Retarder trans, Retardation:'
	print,a(0:2)
        printf,2,' '
        printf,2,'Polarizer trans, Retarder trans, Retardation:'
	printf,2,a(0:2)
	print,'Input Stokes Vector:'
	print,a(3:6)
	printf,2,'Input Stokes Vector:'
	printf,2,a(3:6)

        ; Omatrix:
        ; filling
	o_mx=[[1.,a(7:9)],[1.,a(10:12)],[1.,a(13:15)],[1.,a(16:18)]]
        o_matrix[*,*,ireg]=o_mx        
        ; print
	print,'O Matrix:'
	print,o_mx
	printf,2,'O Matrix:'
        printf,2,o_mx
        
        ; Amatrix:
        ; calculation
        a_mx=transpose(o_mx)##o_mx ;compute A matrix
        a_matrix[*,*,ireg]=a_mx
        ; print        
	print,'A Matrix:'
	print,a_mx
	printf,2,'A Matrix:'
	printf,2,a_mx

        ; Amatrix inverted:
        ; calculation
        a_invert=invert(a_mx,status,/double) ;compute inverse of A
        a_invert_matrix[*,*,ireg]=a_invert
        print        
        print,'A_invert - status (0=success):',status
        printf,2,'A_invert - status (0=success):',status
        print
        printf,2,' '
        print,'A_invert Matrix:'
        print,a_invert
	printf,2,'A_invert Matrix:'
	printf,2,a_invert 

        ; Efficiences
        eff=fltarr(4)
        ; calculation        
	for i=0,3 do eff[i]=1./sqrt(4.*a_invert(i,i))   ;compute efficiencies
        efficiencies[*,ireg]=eff
        ; print        
	print,'Efficiencies: ',eff
	printf,2,'Efficiencies: ',eff

        ; Dmatrix:
        ; calculation
	d_mx=a_invert##transpose(o_mx)			;compute demodulation matrix
	d_matrix[*,*,ireg]=d_mx
        ; print                
	print,'D Matrix:'
	print,d_mx
	printf,2,'D Matrix:'
	printf,2,d_mx

        ; chi2:
        ; calculation
	chi2[ireg]=call_function('min_funct_st_jr',a)
        ; print                
	print,'Chisq: ',chi2[ireg]
	printf,2,'Chisq: ',chi2[ireg]

     endfor ; ireg - line/region

; plots of the basic parameters
str_line=strtrim(string(fix(wave)),2)
if strmid(str_line,0,1) ne 0 then str_line='0'+str_line
sav_out_tmp=date+'_'+str_line+'_set_'+set_str+'_camera_'+cam_str
p,sav_out_tmp

if dev ne 'p' then window,0,tit='Transmisstion, retardance, chi2'
!p.multi=[0,2,2,0,0]
plot,regions,ptrans,xtit='Wavelength (nm)',ytit='Polarizer Transmission',chars=1
oplot,regions,ptrans,psym=1,symsize=6,thick=6
plot,regions,rtrans,xtit='Wavelength (nm)',ytit='Retarder Transmission',chars=1
oplot,regions,rtrans,psym=1,symsize=6,thick=6
plot,regions,delta,xtit='Wavelength (nm)',ytit='Calibration Retardance',chars=1
oplot,regions,delta,psym=1,symsize=6,thick=6
plot,regions,chi2/stokes[0,*]^2,xtit='Wavelength (nm)',ytit='Chi/I^2',chars=1
oplot,regions,chi2/stokes[0,*]^2,psym=1,symsize=6,thick=6
if save_jpg eq 1 then window_2_png_inverted,sav_out_tmp+'_trans_ret_chi2'

if dev ne 'p' then window,1,tit='Stokes I, Q/I, U/I, V/I'
yt=['Input I [counts]','Input Q/I','Input U/I','Input V/I']
plot,regions,stokes[0,*],xtit='Wavelength (nm)',ytit=yt[0]
oplot,regions,stokes[0,*],psym=1,symsize=6,thick=6
for i=1,3 do begin
	plot,regions,stokes[i,*]/stokes[0,*],xtit='Wavelength (nm)',ytit=yt[i]
	oplot,regions,stokes[i,*]/stokes[0,*],psym=1,symsize=6,thick=6
endfor
if save_jpg eq 1 then window_2_png_inverted,sav_out_tmp+'_Stokes_IQUV'

if dev ne 'p' then window,2,tit='Efficiencies'
s=['I','Q','U','V']
for i=0,3 do begin
	plot,regions,efficiencies[i,*],xtit='Wavelength (nm)',ytit='Efficiency',yr=[0,1],tit=s[i]
	oplot,regions,efficiencies[i,*],psym=1,symsize=6,thick=6
endfor
if save_jpg eq 1 then window_2_png_inverted,sav_out_tmp+'_efficiencies'

; saving the basic results
save,regions,stokes,o_matrix,a_matrix,a_invert_matrix,d_matrix,efficiencies,chi2,file=date+fileadd1+'_cal.sav'

!p.multi=0

close,2

stop
;closepost,dev

print,'done'
end
