;+
;  Name: SCD_Create_Cmatrix
;  
;  Description:
;    Procedure to create the Calibration matrix (C matrix) comprised
;    of the set of Stokes vectors output by the calibration optics.
;    We assume that the Mueller matrices of the calibration optics are known.
;    The Mueller matrices will be read in idl .sav format from the
;    optional in_dir directory or the current directory. The 6 .sav
;    files (one for each calibration optic) are named Am.sav where m =
;    1 to 6. The files contain the Mueller matrix as a function of
;    wavelength, and an array of the wavelength (nm). The desired
;    wavelength of the C matrix is also input and the Mueller matrices
;    are interpolated to the desired wavelength. (see SCD Calibration
;    writeup for details and references)
;   
;  Inputs:
;    Ic - the Stokes vector input into the calibration optics (4 vector)
;    wave - the desired wavelength of the C matrix (nm) 
;    
;    MMS - Mueller matrix (1 x 4 x 4 x nlambda) where nlambda = the number of wavelengths
;    L - the wavelength scale (nm)
;    
;  Outputs:
;    Cmatrix - calibration matrix (4 x m) comprised of the m Stokes
;              vectors produced by the calibration
;              optics where m is the number of calibration states
;
;  Keywords:
;    in_dir - directory location of the Mueller matrix files (optional)
;    
;  Examples:
;    SCD_Create_Cmatrix,Ic,656.3D0,Cmatrix
;    SCD_Create_Cmatrix,Ic,656.3D0,Cmatrix,in_dir=in_dir
;    SCD_Create_Cmatrix,Ic,656.3D0,Cmatrix,in_dir=in_dir,plot=plot
;    
;  Author: S. Tomczyk
;
;  Modification History:
;
;  2022/12/17 - J. Rybak: - transposition of the A5 and A6 MM values
;                           to have the values as in the ST's
;                           PDF plots
;                         - plot option to plot MMs~wa wavelength to
;                           JPEG pitures
;-
pro SCD_Create_Cmatrix,Ic,wave,Cmatrix,in_dir=in_dir,plot=plot

debug = 'yes'     ;debug mode ('yes' or 'no')

m = 6                                               ;number of modulation states
Cmatrix = dblarr(4,m)                               ;array to hold Calibration matrix

if keyword_set(in_dir) then dir = in_dir else dir = ''  ;optionally set input data diectory

mm_wave = dblarr(4,4)

for i=0,m-1 do begin
   
  name = string(format='("A",i1,".sav")',i+1)       ;read in Mueller matrix
  restore,file = dir+name   ; -> l(nw),mms(1,4,4,nw)      
  nlambda = n_elements(L)
  mm = double(reform(MMS,4,4,nlambda)) ; -> mm(4,4,nlambda)

  ; transposition of the A5 and A6 MM values to have the values as in the
  ; PDF plots
  if i eq 4 or i eq 5 then begin
     mmt=mm
     for ll=0,nlambda-1 do begin
        mmtmp=reform(mm(*,*,ll))
        mmt(*,*,ll)=transpose(mmtmp)
     endfor
     mm=mmt
  endif
  
  ; plot
  if keyword_set(plot) then begin
     window,0,xs=1500,ys=900,tit='A'+strtrim(string(i+1),2)
     !p.multi=[0,4,4]
     yr=[-1,1] & xr=[500,1100]
     for j=0,3 do begin
        for k=0,3 do begin
           if j ne 3 and k eq 0 then plot,l,mm(j,k,*),$
                                          xr=xr,yr=yr,yticks=5,thick=3,xticks=6,$
                                          ytit='Amplitude', xtickformat="(A1)",$
                                          xminor=4,yminor=4        
           if j ne 3 and k ne 0 then plot,l,mm(j,k,*),$
                                          xr=xr,yr=yr,yticks=5,thick=3,xticks=6,$
                                          ytickformat="(A1)", xtickformat="(A1)",$
                                          xminor=4,yminor=4
           if j eq 3 and k eq 0 then plot,l,mm(j,k,*),$
                                          xr=xr,yr=yr,yticks=5,thick=3,xticks=6,$
                                          ytit='Amplitude',xtit='Wavelength',$
                                          xminor=4,yminor=4
           if j eq 3 and k ne 0 then plot,l,mm(j,k,*),$
                                          xr=xr,yr=yr,yticks=5,thick=3,xticks=6,$
                                          ytickformat="(A1)",xtit='Wavelength',$
                                          xminor=4,yminor=4
           oplot,xr,[0,0]
        endfor
     endfor
     name_jpeg = string(format='("A",i1,"")',i+1)  
     window_2_jpeg,in_dir+name_jpeg
     ;stop
  endif
  
  for j=0,3 do for k=0,3 do begin          ;interpolate matrix to desired wavelength
     mm_wave[j,k] = interpol(mm[j,k,*],L,wave) ; -> mm_wave(4,4)  
  endfor
 
  ; ST
  Cmatrix[*,i] = mm_wave##Ic    ;compute C matrix

endfor

if debug eq 'yes' then begin
   ;print
   print,' C_matrix'
   print
   for i=0,3 do print,format='(6f10.6)',Cmatrix[i,*]
   print
endif

return
end
