;+
;  Name: SCD_Create_Cmatrix
;  
;  Description:
;    Procedure to create the Calibration matrix (C matrix) comprised of the set of Stokes vectors output
;    by the calibration optics. We assume that the Mueller matrices of the calibration optics are known.
;    The Mueller matrices will be read in idl .sav format from the optional in_dir directory or the current
;    directory. The 6 .sav files (one for each calibration optic) are named Am.sav where m = 1 to 6. The 
;    files contain the Mueller matrix as a function of wavelength, and an array of the wavelength (nm). The
;    desired wavelength of the C matrix is also input and the Mueller matrices are interpolated to the desired
;    wavelength. (see SCD Calibration writeup for details and references)
;   
;  Inputs:
;    Ic - the Stokes vector input into the calibration optics (4 vector)
;    wave - the desired wavelength of the C matrix (nm) 
;    
;    MMS - Mueller matrix (1 x 4 x 4 x nlambda) where nlambda = the number of wavelengths
;    L - the wavelength scale (nm)
;    
;  Outputs:
;    Cmatrix - calibration matrix (4 x m) comprised of the m Stokes vectors produced by the calibration
;              optics where m is the number of calibration states
;
;  Keywords:
;    in_dir - directory location of the Mueller matrix files (optional)
;    
;  Examples:
;    SCD_Create_Cmatrix,Ic,656.3D0,Cmatrix
;    SCD_Create_Cmatrix,Ic,656.3D0,Cmatrix,in_dir=in_dir
;    
;  Author: S. Tomczyk
;
;  Modification History:
;
;-
pro SCD_Create_Cmatrix,Ic,wave,Cmatrix,in_dir=in_dir

debug = 'no'     ;debug mode ('yes' or 'no')

m = 6                                               ;number of modulation states
Cmatrix = dblarr(4,m)                               ;array to hold Calibration matrix

if keyword_set(in_dir) then dir = in_dir else dir = ''  ;optionally set input data diectory

mm_wave = dblarr(4,4)

for i=0,m-1 do begin 
  name = string(format='("A",i1,".sav")',i+1)       ;read in Mueller matrix
  restore,file = dir+name         
  nlambda = n_elements(L)
  mm = double(reform(MMS,4,4,nlambda))

  for j=0,3 do for k=0,3 do begin                   ;interpolate matrix to desired wavelength
    mm_wave[j,k] = interpol(mm[j,k,*],L,wave)   
  endfor
  
  Cmatrix[*,i] = mm_wave##Ic                        ;compute C matrix
endfor

if debug eq 'yes' then for i=0,3 do print,format='(6f10.6)',Cmatrix[i,*]

end